<?php

/**
* This file is part of the initpro/kassa-sdk library
*
* For the full copyright and license information, please view the LICENSE
* file that was distributed with this source code.
*/

namespace InitproTest\KassaSdk;

use Initpro\KassaSdk\Agent;
use Initpro\KassaSdk\AuthorisedPerson;
use Initpro\KassaSdk\CalculationMethod;
use Initpro\KassaSdk\CalculationSubject;
use Initpro\KassaSdk\Cashier;
use Initpro\KassaSdk\Check;
use Initpro\KassaSdk\Correction;
use Initpro\KassaSdk\CorrectionCheck;
use Initpro\KassaSdk\Payment;
use Initpro\KassaSdk\Position;
use Initpro\KassaSdk\GroupManager;
use Initpro\KassaSdk\TaxSystem;
use Initpro\KassaSdk\Vat;

class GroupManagerTest extends \PHPUnit_Framework_TestCase
{
    private $client;
    private $qm;

    protected function setUp()
    {
        $this->client = $this
            ->getMockBuilder('\Initpro\KassaSdk\Client')
            ->disableOriginalConstructor()
            ->getMock();
        $this->qm = new GroupManager($this->client);
    }

    public function testRegisterGroup()
    {
        $this->assertFalse($this->qm->hasGroup('my-queue'));
        $this->assertTrue($this->qm->registerGroup('my-queue', 'queue-id')->hasGroup('my-queue'));
    }

    /**
     * @expectedException \InvalidArgumentException
     */
    public function testSetDefaultGroupFailedWithUnregisteredGroup()
    {
        $this->qm->setDefaultGroup('my-queue');
    }

    public function testSetDefaultGroupSucceeded()
    {
        $qm = $this->qm->registerGroup('my-queue', 'queue-id')->setDefaultGroup('my-queue');
        $this->assertEquals($qm, $this->qm);
    }

    /**
     * @expectedException \InvalidArgumentException
     */
    public function testIsGroupActiveFailedWithUnregisteredGroup()
    {
        $this->qm->isGroupActive('my-queue');
    }

    public function testIsGroupActiveTrue()
    {
        $this->qm->registerGroup('my-queue', 'queue-id');
        $this->client
            ->expects($this->once())
            ->method('sendRequest')
            ->with($this->equalTo('lk/api/v1/groups/queue-id'))
            ->willReturn(['state' => 'active']);
        $this->assertTrue($this->qm->isGroupActive('my-queue'));
    }

    public function testIsGroupActiveFalseWithPassive()
    {
        $this->qm->registerGroup('my-queue', 'queue-id');
        $this->client
            ->expects($this->once())
            ->method('sendRequest')
            ->with($this->equalTo('lk/api/v1/groups/queue-id'))
            ->willReturn(['state' => 'passive']);
        $this->assertFalse($this->qm->isGroupActive('my-queue'));
    }

    public function testIsGroupActiveFalseWithNonArray()
    {
        $this->qm->registerGroup('my-queue', 'queue-id');
        $this->client
            ->expects($this->once())
            ->method('sendRequest')
            ->with($this->equalTo('lk/api/v1/groups/queue-id'))
            ->willReturn('string');
        $this->assertFalse($this->qm->isGroupActive('my-queue'));
    }

    /**
     * @expectedException \LogicException
     * @expectedExceptionMessage Default queue is not set
     */
    public function testPutCheckFailedWithoutDefaultGroup()
    {
        $check = $this->getMockBuilder('\Initpro\KassaSdk\Check')->disableOriginalConstructor()->getMock();
        $this->qm->putCheck($check);
    }

    /**
     * @expectedException \InvalidArgumentException
     * @expectedExceptionMessage Unknown queue "my-queue"
     */
    public function testPutCheckFailedWithUnregisteredGroup()
    {
        $check = $this->getMockBuilder('\Initpro\KassaSdk\Check')->disableOriginalConstructor()->getMock();
        $this->qm->putCheck($check, 'my-queue');
    }

    public function testPutCheckToDefaultGroupSucceeded()
    {
        $this->qm->registerGroup('my-queue', 'queue-id');
        $this->qm->setDefaultGroup('my-queue');

        $vat = new Vat('18%');
        $position = new Position('name', 100, 1, 100, 0, $vat);
        $position->setId('123');
        $payment = new Payment(Payment::TYPE_CARD, 100);

        $check = Check::createSell('id', 'user@host', TaxSystem::COMMON);
        $check->addPosition($position);
        $check->addPayment($payment);
        $data = $check->asArray();
        $path = 'lk/api/v1/groups/queue-id/payment';
        $rep = ['key' => 'value'];
        $this->client->expects($this->once())->method('sendRequest')->with($path, $data)->willReturn($rep);

        $this->assertEquals($this->qm->putCheck($check), $rep);
    }

    public function testPutCheckToCustomGroupSucceeded()
    {
        $this->qm->registerGroup('default-queue', 'default-id');
        $this->qm->setDefaultGroup('default-queue');
        $this->qm->registerGroup('my-queue', 'queue-id');

        $vat = new Vat('no');
        $position = (new Position('name', 100, 1, 100, 0, $vat))->setMeasureName('Kg');
        $payment = new Payment(Payment::TYPE_CASH, 100);
        $this->assertEquals($payment->getSum(), 100);

        $check = Check::createSellReturn('id', 'user@host', TaxSystem::COMMON)->setShouldPrint(true);
        $check->addPosition($position);
        $check->addPayment($payment);
        $data = $check->asArray();
        $this->assertTrue($data['print']);
        $path = 'lk/api/v1/groups/queue-id/payment';
        $rep = ['key' => 'val'];
        $this->client->expects($this->once())->method('sendRequest')->with($path, $data)->willReturn($rep);

        $this->assertEquals($this->qm->putCheck($check, 'my-queue'), $rep);
    }

    public function testPutBuyCheckSucceeded()
    {
        $this->qm->registerGroup('my-queue', 'queue-id');
        $this->qm->setDefaultGroup('my-queue');

        $vat = new Vat('18%');
        $position = new Position('name', 100, 1, 100, 0, $vat);
        $position->setId('123');
        $payment = new Payment(Payment::TYPE_CARD, 100);

        $check = Check::createBuy('id', 'user@host', TaxSystem::COMMON);
        $check->addPosition($position);
        $check->addPayment($payment);
        $data = $check->asArray();
        $path = 'lk/api/v1/groups/queue-id/payment';
        $rep = ['key' => 'value'];
        $this->client->expects($this->once())->method('sendRequest')->with($path, $data)->willReturn($rep);

        $this->assertEquals($this->qm->putCheck($check), $rep);
    }

    public function testPutBuyReturnCheckSucceeded()
    {
        $this->qm->registerGroup('my-queue', 'queue-id');
        $this->qm->setDefaultGroup('my-queue');

        $vat = new Vat('18%');
        $position = new Position('name', 100, 1, 100, 0, $vat);
        $position->setId('123');
        $payment = new Payment(Payment::TYPE_CARD, 100);

        $check = Check::createBuyReturn('id', 'user@host', TaxSystem::COMMON);
        $check->addPosition($position);
        $check->addPayment($payment);
        $data = $check->asArray();
        $path = 'lk/api/v1/groups/queue-id/payment';
        $rep = ['key' => 'value'];
        $this->client->expects($this->once())->method('sendRequest')->with($path, $data)->willReturn($rep);

        $this->assertEquals($this->qm->putCheck($check), $rep);
    }

    public function testPutCheckFFD105Succeeded()
    {
        $this->qm->registerGroup('my-queue', 'queue-id');
        $this->qm->setDefaultGroup('my-queue');

        $vat = new Vat('18%');
        $position = new Position('name', 100, 1, 100, 0, $vat);
        $position->setId('123');
        $position->setCalculationMethod(CalculationMethod::FULL_PAYMENT);
        $position->setCalculationSubject(CalculationSubject::PRODUCT);

        $agent = new Agent(Agent::COMMISSIONAIRE, "+77777777777", "ООО 'Лютик'", "12345678901");
        $position->setAgent($agent);

        $check = Check::createSell('id', 'user@host', TaxSystem::COMMON);
        $check->addPosition($position);

        $payment = new Payment(Payment::TYPE_CARD, 100);
        $check->addPayment($payment);

        $cashier = new Cashier('Иваров И.П.', '1234567890123');
        $check->addCashier($cashier);

        $data = $check->asArray();
        $path = 'lk/api/v1/groups/queue-id/payment';
        $rep = ['key' => 'value'];
        $this->client->expects($this->once())->method('sendRequest')->with($path, $data)->willReturn($rep);

        $this->assertEquals($this->qm->putCheck($check), $rep);
    }

    public function testPutSellCorrectionCheckSucceded()
    {
        $this->qm->registerGroup('my-queue', 'queue-id');
        $correction = Correction::createSelf('2012-12-21', '4815162342', 'description');
        $check = CorrectionCheck::createSell('4815162342', '4815162342', TaxSystem::PATENT, $correction);
        $payment = new Payment(Payment::TYPE_CARD, 4815);
        $vat = new Vat('118');
        $check->setPayment($payment, $vat);
        $authorised_person = new AuthorisedPerson('Иваров И.И.', '123456789012');
        $check->setAuthorisedPerson($authorised_person);
        $data = $check->asArray();
        $path = 'lk/api/v1/groups/queue-id/payment';
        $rep = ['key' => 'val'];
        $this->client->expects($this->once())->method('sendRequest')->with($path, $data)->willReturn($rep);
        $this->assertEquals($this->qm->putCheck($check, 'my-queue'), $rep);
    }

    public function testPutSellReturnCorrectionCheckSucceded()
    {
        $this->qm->registerGroup('my-queue', 'queue-id');
        $correction = Correction::createForced('2012-12-21', '4815162342', 'description');
        $check = CorrectionCheck::createSellReturn('4815162342', '4815162342', TaxSystem::PATENT, $correction);
        $payment = new Payment(Payment::TYPE_CARD, 4815);
        $vat = new Vat('118');
        $check->setPayment($payment, $vat);
        $authorised_person = new AuthorisedPerson('Иваров И.И.', '123456789012');
        $check->setAuthorisedPerson($authorised_person);
        $data = $check->asArray();
        $path = 'lk/api/v1/groups/queue-id/payment';
        $rep = ['key' => 'val'];
        $this->client->expects($this->once())->method('sendRequest')->with($path, $data)->willReturn($rep);
        $this->assertEquals($this->qm->putCheck($check, 'my-queue'), $rep);
    }
}
