(function (_, $) {
    const validationTargets = new Map([
        ['ServerUrl', 'addon_option_rus_initpro_kassa_server_url'],
        ['ShopId', 'addon_option_rus_initpro_kassa_shop_id'],
        ['ShopSecret', 'addon_option_rus_initpro_kassa_shop_secret'],
        ['QueueId', 'addon_option_rus_initpro_kassa_queue_id'],

        ['StatusFullPayment', 'addon_option_rus_initpro_kassa_status_full_payment'],
        ['StatusPartialPrepayment', 'addon_option_rus_initpro_kassa_status_partial_prepayment'],
        ['StatusPartialFullPayment', 'addon_option_rus_initpro_kassa_status_partial_full_payment'],
        ['StatusRefund', 'addon_option_rus_initpro_kassa_status_refund'],

    ]);

    const errors = {
        ServerUrl: new Set(),
        ShopId: new Set(),
        ShopSecret: new Set(),
        QueueId: new Set(),
        StatusFullPayment: new Set(),
        StatusPartialPrepayment: new Set(),
        StatusPartialFullPayment: new Set(),
        StatusRefund: new Set(),
    }

    const errorMessages = {
        required: 'Обязательное поле',
        statusUniquenessRequired: 'Статусы должны быть уникальны',
        twoReceiptsPath: 'Обязательное поле при "Формировании двух чеков"'
    }

    function createErrorMessageNode(id) {
        const controlNode = document.querySelector(`[id^=${id}_]`);

        if (!controlNode) return

        const messageNode = document.createElement("div");
        messageNode.classList.add('initpro-kassa-settings--error-message');

        controlNode.after(messageNode);
    }

    function getControlByKey(key) {
        return document.querySelector(`[id^=${validationTargets.get(key)}_]`)
    }

    function addError(controlKey, errorType) {
        errors[controlKey].add(errorType);
    }

    function removeError(controlKey, errorType) {
        errors[controlKey].delete(errorType);
    }

    function showErrorMessage(key, show = true) {
        const control = getControlByKey(key)
        const controlGroup = control.closest('.control-group.rus_initpro_kassa');
        const messageNode = controlGroup.querySelector('.initpro-kassa-settings--error-message');

        const errorTypes = errors[key]
        const iterator = errorTypes.values()
        const errorType = iterator.next().value

        const message = errorMessages[errorType]

        if (message && show) {
            controlGroup.classList.add('hasError');
            messageNode.innerText = message;
        } else {
            controlGroup.classList.remove('hasError');
        }
    }

    function notEmptyRule(key) {
        const control = getControlByKey(key)

        if (!control.value) {
            addError(key, 'required')
            return false
        }

        removeError(key, 'required')
        return true;
    }

    function statusUniquenessRule() {
        let valid = true
        const keys = ['StatusFullPayment', 'StatusPartialPrepayment', 'StatusPartialFullPayment', 'StatusRefund']
        const controls = keys.map(key => getControlByKey(key))

        for (let i = 0; i < controls.length; i++) {
            if (controls[i].value === 'no') {
                removeError(keys[i], 'statusUniquenessRequired')
                continue
            }

            let unique = true

            for (let j = 0; j < controls.length; j++) {
                if (i == j) continue
                if (controls[i].value === controls[j].value) {
                    addError(keys[j], 'statusUniquenessRequired')
                    unique = false
                }
            }

            if (unique) {
                removeError(keys[i], 'statusUniquenessRequired')
            } else {
                addError(keys[i], 'statusUniquenessRequired')
                valid = false
            }
        }

        return valid
    }

    function twoReceiptsPathRule() {
        let firstControl = getControlByKey('StatusPartialPrepayment')
        let secondControl = getControlByKey('StatusPartialFullPayment')

        let firstValue = firstControl.value
        let secondValue = secondControl.value

        let isValid = true
        if (
            (firstValue === 'no' && secondValue === 'no')
            || (firstValue !== 'no' && secondValue !== 'no')
        ) {
            removeError('StatusPartialPrepayment', 'twoReceiptsPath')
            removeError('StatusPartialFullPayment', 'twoReceiptsPath')
        } else if (firstValue === 'no' && secondValue !== 'no') {
            addError('StatusPartialPrepayment', 'twoReceiptsPath')
            isValid = false
        } else if (firstValue !== 'no' && secondValue === 'no') {
            addError('StatusPartialFullPayment', 'twoReceiptsPath')
            isValid = false
        }

        return isValid
    }

    function showOpenStatusAlert() {
        const keys = ['StatusFullPayment', 'StatusPartialPrepayment', 'StatusPartialFullPayment', 'StatusRefund']

        keys.forEach(key => {
            const control = getControlByKey(key)

            if (errors[key].size !== 0) return

            const controlGroup = control.closest('.control-group.rus_initpro_kassa');
            const messageNode = controlGroup.querySelector('.initpro-kassa-settings--error-message');

            if (control.value === 'O') {
                controlGroup.classList.add('status-warning');
                messageNode.innerText = 'Не рекомендуется использовать системный статус "Открыт".';
            } else {
                controlGroup.classList.remove('status-warning');
            }
        })
    }

    function applyRules(key) {
        const control = getControlByKey(key);
        if (!control) return;

        if (['ServerUrl', 'ShopId', 'ShopSecret', 'QueueId'].includes(key)) {
            control.addEventListener('blur', () => {
                notEmptyRule(key)
                showErrorMessage(key)
            });

            control.addEventListener('input', () => {
                notEmptyRule(key)
                showErrorMessage(key)
            })
        }

        if (['StatusPartialPrepayment', 'StatusPartialFullPayment', 'StatusFullPayment', 'StatusRefund'].includes(key)) {
            control.addEventListener('change', () => {
                statusUniquenessRule()
                twoReceiptsPathRule()

                showErrorMessage('StatusPartialPrepayment')
                showErrorMessage('StatusPartialFullPayment')
                showErrorMessage('StatusFullPayment')
                showErrorMessage('StatusRefund')

                showOpenStatusAlert()
            })
        }
    }

    function prepareValidation() {
        for (let key of validationTargets.keys()) {
            createErrorMessageNode(validationTargets.get(key));
            applyRules(key);
        }

        showOpenStatusAlert();
    }

    function validateForm() {
        let valid = true
        let keys = ['ServerUrl', 'ShopId', 'ShopSecret', 'QueueId']

        keys.forEach(key => {
            valid = notEmptyRule(key) && valid

            showErrorMessage(key)
        })

        keys = ['StatusPartialPrepayment', 'StatusPartialFullPayment', 'StatusFullPayment', 'StatusRefund']
        keys.forEach(key => {
            valid = statusUniquenessRule() && valid
            valid = twoReceiptsPathRule() && valid

            showErrorMessage(key)
        })
        return valid
    }

    $(document).ready(function () {
        prepareValidation();

        $('[data-ca-target-form=update_addon_rus_initpro_kassa_form]').click((e) => {
            if (validateForm()) return

            e.stopImmediatePropagation()
        })
    });
})(Tygh, Tygh.$);
